<?php

namespace App\Models;

use App\Models\Clas;
use App\Models\Country;
use App\Models\Traits\Pointable;
use App\Traits\EncryptationId;
use Laravel\Passport\HasApiTokens;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Spatie\Permission\Traits\HasRoles;
use App\Helpers\ParseUrl;
use Illuminate\Support\Facades\Log;


class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, Pointable, EncryptationId, HasRoles;
    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    // protected $guard_name = 'api';
    protected $guarded = [];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        'pivot'
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'expiration_date' => 'datetime',
        'expiration_membership_date' => 'datetime',
    ];

    protected $appends = [
        'fullName',
        'LeftPoints',
        'RightPoints',
        'active',
        'qualified',
        'encid',
        'membershipActive',
    ];
    
    protected $table= 'users';

    public function getPhotoAttribute($value)
    {
        return  ParseUrl::contacAtrrS3($value);
    }

    public function getfullNameAttribute()
    {
        return $this->name . ' ' . $this->last_name;
    }

    public function getActiveAttribute()
    {
        $account_type = $this->accountType;
        if($account_type->account == "Basic" || $account_type->account == "Guest"){
            return false;
        }
        //Si la fecha de expiracion es mayor a la fecha actual
        $expiro = $this->expiration_date > now() ? true : false;
        $aceptado = $this->request == 2 ? true : false;

        return ($expiro && $aceptado) ? true : false;
    }

    public function getMembershipActiveAttribute(){
        $account_type = $this->accountType;
        if($account_type->account == "Basic" || $account_type->account == "Guest"){
            return false;
        }
        $expiration = $this->expiration_membership_date > now() ? true : false;
        $accepted = $this->request == 2 ? true : false;

        return ($expiration && $accepted) ? true : false;
    }

    public function getQualifiedAttribute(): bool
    {
        $qualified = false;

        $left = $this->classifiedSponsor()->where('position', 0)->with('user')->get()->filter(function ($key) {
            return $key->user->active  == true && $key->user->membershipActive && $key->user->id_account_type != 5 && $key->user->id_account_type != 6;
        })->count();
        $right = $this->classifiedSponsor()->where('position', 1)->with('user')->get()->filter(function ($key) {
            return $key->user->active  == true && $key->user->membershipActive && $key->user->id_account_type != 5 && $key->user->id_account_type != 6;
        })->count();

        if ($left > 0 && $right > 0) {
            $qualified = true;
        }
        return $qualified;
    }


    public function scopeIsActive($query)
    {
        return $query->where('expiration_date', '>', now())->where('request', 2);
    }


    public function country(): BelongsTo
    {
        return $this->belongsTo(Country::class, 'id_country');
    }

    public function sponsor(): BelongsTo
    {
        return $this->belongsTo(User::class, 'id_referrer_sponsor');
    }


    public function paymentsClient(): HasOne
    {
        return $this->hasOne(Payment::class, 'user_id');
    }

    public function paymentsSponsor(): HasMany
    {
        return $this->hasMany(Payment::class, 'id_user_sponsor');
    }

   

    public function accountType(): BelongsTo
    {
        return $this->belongsTo(AccountType::class, 'id_account_type');
    }

    public function documentType(): BelongsTo
    {
        return $this->belongsTo(DocumentType::class, 'id_document_type');
    }

    public function wallets(): HasMany
    {
        return $this->hasMany(Wallet::class);
    }


    public function classifiedSponsor(): HasMany
    {
        return $this->hasMany(Classified::class, 'id_user_sponsor', 'id');
    }

    public function classifiedClients(): HasMany
    {
        return $this->hasMany(Classified::class, 'user_id', 'id');
    }

    public function scopeMyClients($query,$id)
    {
        return $query->where('id_referrer_sponsor', $id);
    }

    //Relacion uno a muchos sponsorLInk
    public function SponsorLink(){
        return $this->hasMany(SponsorLink::class);
    }

    public function scopeQualifiedsAndActive($query)
    {
        return $query->with('accountType')->whereNotIn('id_account_type', ['5', '6', '1'])->get()->filter(function ($key) {
            return $key->qualified == true && $key->active == true && $key->membershipActive == true;
        });
    }

    /**
     * Get all of the courses for the User
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function courses(): HasMany
    {
        return $this->hasMany(Course::class);
    }
    public function scopeMyCourses()
    {
        return $this->courses()->select('users.name','courses.id','courses.title','courses.price','courses.status','courses.id_categories','courses.created_at','courses.description','courses.portada', 'courses.url_portada')->join("users","courses.user_id","=","users.id")->orderBy('courses.created_at','ASC');
    }
    /**
     * The lessons that belong to the User
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function lessons(): BelongsToMany
    {
        return $this->belongsToMany(Clas::class, 'class_users');
    }

    public function purchaseds(): HasMany{
        return $this->hasMany(PurchasedCourse::class);
    }

    // $user->lessons()->attach(2);
    // $user->lessons()->dettach(2);
    // $user->lessons()->sync(2);
    // $user->lessons()->syncWithoutDetaching(2);
    public function messages():HasMany
    {
        return $this->hasMany(Message::class,'id_user_receiver');
    }
    public function jsPermissions()
    {
        return json_encode([
                'roles' => $this->getRoleNames(),
                'permissions' => $this->getAllPermissions()->pluck('name'),
            ]);
    }
}
