<?php

namespace App\Http\Controllers;

use App\Models\Classified;
use App\Models\Option;
use App\Models\BinaryCutHistory;
use App\Models\Payment;
use App\Models\Point;
use App\Models\RankBinary;
use App\Models\RankBonus;
use App\Models\User;
use App\Models\Wallet;
use App\Models\WalletMovements;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;


class BinaryCutController extends Controller
{
    public function __construct()
    {
        $this->middleware('can:binarycut.index');
    }

    public function index()
    {
        $this->authorize('viewAny', auth()->user());
        $users = User::qualifiedsAndActive();

        // Obtener los rangos para mostrar los límites en la vista
        $ranks = RankBonus::select('id', 'vol_min', 'pack_max', 'active_direct', 'max_pay', 'monthly_bonus')
            ->get();

        return view('content.binarycut.index', compact('users'));
    }

    public function store()
    {
        $this->authorize('viewAny', auth()->user());
        Log::info('Iniciando el proceso de store');

        try {
            DB::beginTransaction();

            $users = User::qualifiedsAndActive();
            Log::info('Usuarios obtenidos: ' . $users->count());

            $ranks = RankBonus::select('id', 'vol_min', 'pack_max', 'active_direct', 'max_pay', 'monthly_bonus')
                ->get();
            Log::info('Rangos obtenidos: ' . $ranks->count());

            $batch = Option::where('description', 'batch')->first();
            Log::info('Batch obtenido: ' . $batch->value);

            $last_batch = (int)$batch->value;

            foreach ($users as $user) {
                Log::info('Procesando usuario ID: ' . $user->id);

                // Cálculo de puntos
                $maxPoints = max($user->LeftPoints, $user->RightPoints);
                $minPoints = min($user->LeftPoints, $user->RightPoints);
                $sideMax = $user->LeftPoints > $user->RightPoints ? 0 : 1;

                // Obtener rango actual
                $currentRank = RankBinary::where('user_id', $user->id)
                    ->orderBy('batch', 'desc')
                    ->first();

                if (!$currentRank) {
                    $currentRank = RankBonus::where('id', 1)->first();
                } else {
                    $currentRank = RankBonus::find($currentRank->rank_id);
                }

                // Asignar nuevo rango
                $my_rank = $this->setRanks($user->id, $minPoints, $ranks, $last_batch);

                // Actualizar puntos
                $user->points()->where('status', 1)->update(['status' => 0]);
                $user->points()->create([
                    'user_id' => $user->id,
                    'points' => $maxPoints - $minPoints,
                    'side' => $sideMax,
                    'reason' => "Binary cut"
                ]);

                // Procesar wallet
                $myWallet = Wallet::where('user_id', $user->id)->first();
                $maxTransfer = $my_rank->max_pay;
                $amountToTransfer = ($minPoints * 1) * ($user->accountType->pay_in_binary / 100);

                // Limitar al máximo permitido
                if ($amountToTransfer > $maxTransfer) {
                    $amountToTransfer = $maxTransfer;
                }

                // Crear movimiento de wallet si hay monto a transferir
                if ($amountToTransfer > 0) {
                    $movement = new WalletMovements();
                    $movement->wallet_id = $myWallet->id;
                    $movement->amount = $amountToTransfer;
                    $movement->type = 1;
                    $movement->reason = 'Bono binario';
                    $movement->batch = $last_batch;
                    $movement->bonus_type_id = 4;
                    $movement->save();
                }

                // Guardar historial del corte
                BinaryCutHistory::create([
                    'user_id' => $user->id,
                    'rank_id' => $my_rank->id,
                    'left_points' => $user->LeftPoints,
                    'right_points' => $user->RightPoints,
                    'transferred_amount' => $amountToTransfer,
                    'batch' => $last_batch
                ]);

                $user->wallets('status', 1)->update(['status' => 2]);
            }

            // Procesar bonos por rangos
            foreach ($users as $user) {
                $myWallet = Wallet::where('user_id', $user->id)->first();
                $myRank = RankBinary::join('rank_bonus', 'rank_binary.rank_id', '=', 'rank_bonus.id')
                    ->where('rank_binary.user_id', $user->id)
                    ->where('rank_binary.batch', $last_batch)
                    ->select('rank_bonus.id', 'rank_bonus.max_pay')
                    ->first();

                if ($myRank->id != 1) {
                    $this->rankBonusPay($user, $last_batch, $myWallet, $myRank);
                }
            }

            // Actualizar batch
            $last_batch++;
            $batch->value = $last_batch;
            $batch->save();

            DB::commit();

            return redirect()->route('binarycut.index')
                ->withSuccess('Binary cut successfully');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Error en el proceso de corte binario: ' . $e->getMessage());
            return redirect()->route('binarycut.index')
                ->withError('Error processing binary cut: ' . $e->getMessage());
        }
    }

    public function myHistory()
    {
        $histories = BinaryCutHistory::where('user_id', auth()->id())
            ->with(['rank'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return response()->json($histories);
    }

    public function userHistory($userId)
    {
        $this->authorize('viewAny', auth()->user());

        $histories = BinaryCutHistory::where('user_id', $userId)
            ->with(['rank', 'user'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('content.binarycut.user-history', compact('histories'));
    }

    public function getDirects($user, $batch)
    {
        $directs = User::join('wallet', 'users.id', '=', 'wallet.user_id')
            ->join('wallet_movements', 'wallet.id', '=', 'wallet_movements.wallet_id')
            ->where('users.id_referrer_sponsor', $user->id)
            ->where('wallet_movements.batch', $batch)
            ->where('wallet_movements.bonus_type_id', 4)
            ->select('users.id', 'users.username', 'wallet_movements.amount', 'wallet.id')
            ->get();
        return $directs;
    }

    public function deliverBonusRank($total_bonus, $myWallet, $batch, $i, $bonus_percentage, $max_pay)
    {
        $mount_to_pay = $total_bonus * $bonus_percentage / 100;
        if ($mount_to_pay != 0) {
            if ($mount_to_pay >= $max_pay) {
                $wallet_movement = new WalletMovements();
                $wallet_movement->wallet_id = $myWallet->id;
                $wallet_movement->amount = $max_pay;
                $wallet_movement->type = 1;
                $wallet_movement->status = 1;
                $wallet_movement->reason = "Bono generacional, " . $i . "° generación";
                $wallet_movement->bonus_type_id = 5;
                $wallet_movement->batch = $batch;
                $wallet_movement->save();
            } else {
                $wallet_movement = new WalletMovements();
                $wallet_movement->wallet_id = $myWallet->id;
                $wallet_movement->amount = $mount_to_pay;
                $wallet_movement->type = 1;
                $wallet_movement->status = 1;
                $wallet_movement->reason = "Bono generacional, " . $i . "° generación";
                $wallet_movement->bonus_type_id = 5;
                $wallet_movement->batch = $batch;
                $wallet_movement->save();
            }
        }
    }

    public function repeatProcess($directs, $batch, $myWallet, $i, $bonus_percentage, $max_pay)
    {
        $directsAux = [];
        $total_bonus = 0;
        foreach ($directs as $direct) {
            $users = $this->getDirects($direct, $batch);
            array_push($directsAux, $users);
            foreach ($users as $user) {
                $total_bonus += $user->amount;
            }
        }
        $this->deliverBonusRank($total_bonus, $myWallet, $batch, $i, $bonus_percentage, $max_pay);
        return $directsAux;
    }

    //rankBonusPay
    public function rankBonusPay($user, $batch, $myWallet, $myRank)
    {
        $rank = $myRank->id;
        $max_pay = $myRank->max_pay;
        $generation = 0;
        $i = 0;
        $directs = [];
        $bonus_percentage = 0;
        while ($generation != $rank) {
            //primera generacion
            if ($generation == 0) {
                $i = 1;

                // Consulta a la base de datos para obtener el bonus_percentage dinámicamente
                $bonusData = DB::table('generational_bonuses')
                    ->select('g_1', 'range_name')
                    ->where('id', 1)
                    ->first();

                // Validar si se encontró el registro
                if ($bonusData) {
                    $bonus_percentage = $bonusData->g_1; // Aquí el porcentaje se ajusta dinámicamente
                    $range_name = $bonusData->range_name;
                } else {
                    $bonus_percentage = 0; // Valor por defecto si no hay datos
                }

                $total_bonus = 0;
                $directs = $this->getDirects($user, $batch);

                foreach ($directs as $direct) {
                    $total_bonus += $direct->amount;
                }

                $this->deliverBonusRank($total_bonus, $myWallet, $batch, $i, $bonus_percentage, $max_pay);
                $generation++;
            } else {
                //segunda generacion
                $i = 2;
                $bonus_percentage = 5;
                $directs2 = [];
                if (!empty($directs)) {
                    $directs2 = $this->repeatProcess($directs, $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;

                if ($rank == $generation) {
                    break;
                }

                //tercera generacion
                $i = 3;
                $bonus_percentage = 5;
                $directs3 = [];
                if (!empty($directs2)) {
                    $directs3 = $this->repeatProcess($directs2[0], $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;
                if ($rank == $generation) {
                    break;
                }

                //cuarta generacion
                $i = 4;
                $bonus_percentage = 3;
                $directs4 = [];
                if (!empty($directs3)) {
                    $directs4 = $this->repeatProcess($directs3[0], $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;
                if ($rank == $generation) {
                    break;
                }

                //quinta generacion
                $i = 5;
                $bonus_percentage = 2;
                $directs5 = [];
                if (!empty($directs4)) {
                    $directs5 = $this->repeatProcess($directs4[0], $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;
                if ($rank == $generation) {
                    break;
                }

                //sexta generacion
                $i = 6;
                $bonus_percentage = 1;
                $directs6 = [];
                if (!empty($directs5)) {
                    $directs6 = $this->repeatProcess($directs5[0], $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;
                if ($rank == $generation) {
                    break;
                }

                //septima generacion
                $i = 7;
                $bonus_percentage = 1;
                $directs7 = [];
                if (!empty($directs6)) {
                    $directs7 = $this->repeatProcess($directs6[0], $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;
                if ($rank == $generation) {
                    break;
                }

                //octava generacion
                $i = 8;
                $bonus_percentage = 1;
                if (!empty($directs7)) {
                    $this->repeatProcess($directs7[0], $batch, $myWallet, $i, $bonus_percentage, $max_pay);
                } else {
                    break;
                }
                $generation++;
                if ($rank == $generation) {
                    break;
                }
                //==========================================================
                //========= codigo para reutilización en proceso============
                //==========================================================

                // while ($generation != $rank) {
                //     $i++;
                //     // $directs = [];
                //     if(!empty($directs)){
                //         if($generation == 1){
                //             $directs2 = $this->repeatProcess($directs, $batch, $myWallet, $i);
                //         }else{
                //             $directs2 = $this->repeatProcess($directs[0], $batch, $myWallet, $i);
                //         }
                //     }else{
                //         break;
                //     }
                //     $directs = $directs2;
                //     $generation++;
                // }
            }
        }
    }

    public function expansionBonus($user_id, $wallet_id, $last_batch)
    {

        // $payments = Payment::where('id_user_sponsor',$user_id)->where('ex_bonus',0)->get();
        $payments = Payment::join('users', 'users.id', '=', 'payments.user_id')
            ->join('account_type', 'account_type.id', '=', 'users.id_account_type')
            ->join('account_type_points_money', 'account_type_points_money.account_type_id', '=', 'users.id_account_type')
            ->where('id_user_sponsor', $user_id)->where('ex_bonus', 0)->whereBetween('account_type.id', [2, 4])
            ->selectRaw('count(account_type_points_money.account_type_id) as n_membership, account_type.account,account_type.id,account_type.price')
            ->groupBy('account_type_points_money.account_type_id')
            ->get();

        $movement = new WalletMovements();
        $movement->wallet_id = $wallet_id;
        $movement->type = 1;
        $movement->reason = 'Expansion Bonus';
        $movement->batch = $last_batch;

        for ($i = 0; $i < sizeof($payments); $i++) {
            if ($payments[$i]->id == 2 || $payments[$i]->id == 3) {
                if ($payments[$i]->n_membership == 4) {

                    $movement->amount = ($payments[$i]->price * 4) * (0.05);
                    $movement->save();
                }
                if ($payments[$i]->n_membership == 5) {
                    $movement->amount = ($payments[$i]->price * 5) * (0.06);
                    $movement->save();
                }
                if ($payments[$i]->n_membership == 6) {
                    $movement->amount = ($payments[$i]->price * 6) * (0.07);
                    $movement->save();
                }
                if ($payments[$i]->n_membership >= 7) {
                    $movement->amount = ($payments[$i]->price * 7) * (0.08);
                    $movement->save();
                }
            };

            if ($payments[$i]->id == 4) {
                if ($payments[$i]->n_membership == 4) {

                    $movement->amount = ($payments[$i]->price * 4) * (0.07);
                    $movement->save();
                }
                if ($payments[$i]->n_membership == 5) {
                    $movement->amount = ($payments[$i]->price * 5) * (0.08);
                    $movement->save();
                }
                if ($payments[$i]->n_membership == 6) {
                    $movement->amount = ($payments[$i]->price * 6) * (0.09);
                    $movement->save();
                }
                if ($payments[$i]->n_membership >= 7) {
                    $movement->amount = ($payments[$i]->price * 7) * (0.1);
                    $movement->save();
                }
            }
        }

        $payments = Payment::where('id_user_sponsor', $user_id)->where('ex_bonus', 0)
            ->update(['ex_bonus' => 1]);
    }

    public function rankBonus($user_id, $vol_min, $ranks)
    {
        $n_active_directs = User::where('id_referrer_sponsor', $user_id)
            ->where('expiration_date', '>', now())
            ->where('expiration_membership_date', '>', now())
            ->count();
        $rank_condition = null;
        foreach ($ranks as $rank) {
            if ($vol_min >= $rank->vol_min && $n_active_directs >= $rank->active_direct) {
                $rank_condition =  $rank;
            }
        }

        $university_count = 0;
        $condition = 0;
        $end_loop = 0;
        $array_users = []; //aqui se almacenan el id de los usuarios de cada nivel
        array_push($array_users, $user_id);
        while ($end_loop == 0 || $condition == 0) {
            $array_users_aux = []; //auxiliar para almacenar los id de cada nivel por pierna
            foreach ($array_users as $array_user) {
                $array_users_aux2 = []; //auxiliar para consultar la cantidad de university por pierna
                $users_bellow = Classified::where('user_id', $array_user)
                    ->select('user_position_left', 'user_position_right')
                    ->first();

                if ($users_bellow->user_position_left != null) {
                    array_push($array_users_aux, $users_bellow->user_position_left);
                    array_push($array_users_aux2, $users_bellow->user_position_left);
                }
                if ($users_bellow->user_position_right != null) {
                    array_push($array_users_aux, $users_bellow->user_position_right);
                    array_push($array_users_aux2, $users_bellow->user_position_right);
                }

                //contador de membresias university
                if (!empty($array_users_aux)) {
                    foreach ($array_users_aux2 as $array_user_aux2) {
                        $membership_uni = User::where('id', $array_user_aux2)
                            ->where('expiration_date', '>', now())
                            ->where('expiration_membership_date', '>', now())
                            ->where('id_account_type', 4)
                            ->count();
                        $university_count += $membership_uni;
                    }
                } else {
                    break 2;
                }
                //se cumple la condicion
                if ($university_count >= $rank_condition->pack_max) {
                    $condition = 1;
                    break 2;
                }
            }
            //si el array_aux esta vacio entonces ya no hay datos para evaluar
            if (empty($array_users_aux)) {
                $end_loop = 1;
            } else {
                $array_users = [];
                $array_users = $array_users_aux;
            }
        }
        if ($condition) {
            return $rank_condition;
        } else {
            foreach ($ranks as $rank) {
                if ($university_count >= $rank->pack_max) {
                    return $rank;
                }
            }
        }

        //return por defecto rector
        return $ranks[count($ranks) - 1];
    }

    public function setRanks($user_id, $vol_min, $ranks, $last_batch)
    {
        //eliminar registro si es q no llega ni al primer rango
        $my_rank = $this->rankBonus($user_id, $vol_min, $ranks);
        $new_rank = new RankBinary();
        $new_rank->user_id = $user_id;
        $new_rank->rank_id = $my_rank->id;
        $new_rank->batch = $last_batch;
        $new_rank->save();
        return $my_rank;
    }

    public function paymentsGeneration($user_id, $ranks, $wallet_id, $last_batch)
    {
        $generation = 1;
        $total_generation = 0;
        $rank = RankBinary::join('rank_bonus', 'rank_bonus.id', '=', 'rank_id')
            ->where('user_id', $user_id)->orderBy('rank_binary.created_at', 'desc')
            ->select('rank_bonus.name', 'rank_bonus.pack_max', 'rank_bonus.active_direct', 'rank_bonus.max_pay')
            ->get()->first();

        //first generation
        $users = User::where('id_referrer_sponsor', $user_id)->where('expiration_date', '>', now())->pluck('id');
        for ($i = 0; $i < count($users); $i++) {
            $pay_son = $this->paymentRoot($users[$i], $ranks);
            $total_generation = $total_generation + $pay_son;
        }

        $movement = new WalletMovements();
        $movement->wallet_id = $wallet_id;
        $movement->amount = $total_generation * 0.05;
        $movement->type = 1;
        $movement->batch = $last_batch;
        $movement->reason = 'Pay 1 Generation';
        $movement->save();

        //2,3,4,5,6, etc.
        while (count($users) > 0) {
            $generation = $generation + 1;
            if ($rank->limit_generation < $generation) {
                break;
            }
            $total_generation = 0;
            $users = User::whereIn('id_referrer_sponsor', $users)->where('expiration_date', '>', now())->pluck('id');
            if ($generation >= 2 && $generation <= 3) {
                for ($i = 0; $i < count($users); $i++) {
                    $pay_son = $this->paymentRoot($users[$i], $ranks);
                    $total_generation = $total_generation + $pay_son;
                }

                $movement = new WalletMovements();
                $movement->wallet_id = $wallet_id;
                $movement->amount = $total_generation * 0.05;
                $movement->type = 1;
                $movement->batch = $last_batch;
                $movement->reason = 'Pay ' . $generation . ' Generation';
                $movement->save();
            } elseif ($generation == 4) {
                for ($i = 0; $i < count($users); $i++) {
                    $pay_son = $this->paymentRoot($users[$i], $ranks);
                    $total_generation = $total_generation + $pay_son;
                }

                $movement = new WalletMovements();
                $movement->wallet_id = $wallet_id;
                $movement->amount = $total_generation * 0.03;
                $movement->type = 1;
                $movement->batch = $last_batch;
                $movement->reason = 'Pay ' . $generation . ' Generation';
                $movement->save();
            } elseif ($generation == 5) {
                for ($i = 0; $i < count($users); $i++) {
                    $pay_son = $this->paymentRoot($users[$i], $ranks);
                    $total_generation = $total_generation + $pay_son;
                }

                $movement = new WalletMovements();
                $movement->wallet_id = $wallet_id;
                $movement->amount = $total_generation * 0.02;
                $movement->type = 1;
                $movement->batch = $last_batch;
                $movement->reason = 'Pay ' . $generation . ' Generation';
                $movement->save();
            } else {
                for ($i = 0; $i < count($users); $i++) {
                    $pay_son = $this->paymentRoot($users[$i], $ranks);
                    $total_generation = $total_generation + $pay_son;
                }

                $movement = new WalletMovements();
                $movement->wallet_id = $wallet_id;
                $movement->amount = $total_generation * 0.01;
                $movement->type = 1;
                $movement->batch = $last_batch;
                $movement->reason = 'Pay ' . $generation . ' Generation';
                $movement->save();
            }
        }
    }

    public function paymentRoot($id_son, $ranks)
    {
        $points_son_left = Point::where('user_id', $id_son)->where('status', 1)->where('side', 0)->sum('points');
        $points_son_rigth = Point::where('user_id', $id_son)->where('status', 1)->where('side', 1)->sum('points');

        $minPoints = 0;
        if ($points_son_left > $points_son_rigth) {
            $minPoints = $points_son_rigth;
        } else {
            $minPoints = $points_son_left;
        }
        $rank_user = $this->rankBonus($id_son, $minPoints, $ranks);
        if ($rank_user->max_pay < $minPoints) {
            $minPoints = $minPoints - ($minPoints - $rank_user->max_pay);
        }

        return $minPoints;
    }
}
