<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\SponsorLink;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Resources\SponsorLinkResource;

class ShareLinkController extends Controller
{
    public function index()
    {
        return view('content.config.share-link');
    }

    public function Add(Request $request)
    {
        try {
            DB::statement('CALL delete_expired_records_procedure()');

            $userTimezone = auth()->user()->timezone ?? 'UTC';
            Log::info('Timezone utilizada', [
                'timezone' => $userTimezone,
                'user_id' => auth()->id()
            ]);

            $userDate = Carbon::now($userTimezone);

            $sponsorLink = new SponsorLink();
            $sponsorLink->user_id = $request->user_id;
            $sponsorLink->url = $request->url . '/' . time();
            $sponsorLink->estado = $request->estado;
            $sponsorLink->fecha_inicio = $userDate;
            $sponsorLink->fecha_fin = $userDate->copy()->addMinutes(15);

            Log::info('Sponsor Link creado', [
                'fecha_inicio' => $sponsorLink->fecha_inicio,
                'fecha_fin' => $sponsorLink->fecha_fin
            ]);

            $result = new SponsorLinkResource($sponsorLink);

            if ($sponsorLink->save()) {
                $tiempoRestanteEnSegundos = now()->diffInSeconds($sponsorLink->fecha_fin);

                Log::info('Tiempo restante calculado', [
                    'tiempoRestanteEnSegundos' => $tiempoRestanteEnSegundos
                ]);

                return response()->json([
                    'resource' => $result,
                    'tiempoRestanteEnSegundos' => max(0, $tiempoRestanteEnSegundos),
                    'fechaFin' => $sponsorLink->fecha_fin->toIso8601String()
                ], 200);
            }

            return response()->json(['error' => 'Error al guardar el enlace'], 400);
        } catch (\Exception $e) {
            Log::error('Error al crear enlace', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json(['error' => 'Error interno del servidor: ' . $e->getMessage()], 500);
        }
    }

    public function obtenerTiempoRestante()
    {
        $sponsorLink = SponsorLink::where('user_id', auth()->id())->latest()->first();

        Log::info('Sponsor Link:', [
            'user_id' => auth()->id(),
            'sponsorLink' => $sponsorLink ? $sponsorLink->toArray() : null,
            'now' => Carbon::now(),
            'timezone' => auth()->user()->timezone
        ]);

        if (!$sponsorLink || Carbon::now()->gt($sponsorLink->fecha_fin)) {
            return response()->json(['tiempoRestanteEnSegundos' => 0]);
        }

        $tiempoRestanteEnSegundos = now()->diffInSeconds($sponsorLink->fecha_fin);

        Log::info('Tiempo restante:', [
            'tiempoRestanteEnSegundos' => $tiempoRestanteEnSegundos,
            'fecha_fin' => $sponsorLink->fecha_fin,
            'now' => Carbon::now()
        ]);

        return response()->json([
            'tiempoRestanteEnSegundos' => max(0, $tiempoRestanteEnSegundos)
        ]);
    }

    public function Delete($id)
    {
        $shareLink = SponsorLink::findOrFail($id);

        if ($shareLink == null) {
            return response()->json([
                'success' => false,
                'message' => 'Ocurrio un error al ejecutar la operación',
                'state' => 404
            ], 404);
        }

        if ($shareLink->delete()) {
            return response()->json([
                'success' => true,
                'message' => 'Registro eliminado con exito',
                'state' => 200
            ], 200);
        };

        return response()->json([
            'success' => false,
            'message' => 'Ocurrio un error',
            'state' => 400
        ], 400);
    }

    public function retornarVista($username)
    {
        $user = User::where('username', $username)->first();
        $link = SponsorLink::where('user_id', $user->id)->latest()->first();
        if (is_null($link) || Carbon::now()->gt($link->fecha_fin)) {
            $link = 0;
        }
        $this->authorize('view', $user);
        return response()->json([$user, $link], 200);
    }
}
