# Configuración de Autenticación con Passport

## Problema: Error "Unauthenticated."

El error ocurre cuando el frontend no envía correctamente el token de autenticación.

## Solución Implementada

### 1. ✅ CORS Corregido
Se ha actualizado el middleware CORS (`app/Http/Middleware/Cors.php`) para:
- Aceptar credenciales correctamente
- Permitir headers de autorización
- Soportar múltiples orígenes locales

### 2. Flujo Correcto de Autenticación

#### Step 1: Login
**Endpoint:** `POST /api/v1/public/auth/login`
```json
{
  "email": "usuario@example.com",
  "password": "password123"
}
```

**Respuesta:**
```json
{
  "message": "Correct login",
  "status": 200,
  "data": {
    "access_token": "eyJ0eXAiOiJKV1QiLCJhbGc...",
    "token_type": "Bearer",
    "user": {
      "id": 1,
      "name": "Usuario",
      "email": "usuario@example.com"
    },
    "role": "admin"
  }
}
```

#### Step 2: Guardar el Token
El frontend debe guardar el `access_token` en `localStorage` o `sessionStorage`:

**JavaScript/React:**
```javascript
// Guardar token
const response = await fetch('http://localhost:8000/api/v1/public/auth/login', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json'
  },
  credentials: 'include',
  body: JSON.stringify({
    email: 'user@example.com',
    password: 'password123'
  })
});

const data = await response.json();
localStorage.setItem('access_token', data.data.access_token);
```

#### Step 3: Enviar el Token en Requests Autenticados
Todas las rutas protegidas requieren el header:
```
Authorization: Bearer <access_token>
```

**JavaScript/React:**
```javascript
const token = localStorage.getItem('access_token');

const response = await fetch('http://localhost:8000/api/v1/course/list', {
  method: 'GET',
  headers: {
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${token}`,
    'Accept': 'application/json'
  },
  credentials: 'include'
});

const data = await response.json();
console.log(data);
```

**Con Axios:**
```javascript
import axios from 'axios';

// Configurar el interceptor
axios.interceptors.request.use((config) => {
  const token = localStorage.getItem('access_token');
  if (token) {
    config.headers['Authorization'] = `Bearer ${token}`;
  }
  return config;
});

// Usar normalmente
const response = await axios.get('http://localhost:8000/api/v1/course/list');
```

**Con Fetch + Helper:**
```javascript
const apiCall = async (url, method = 'GET', body = null) => {
  const token = localStorage.getItem('access_token');
  const headers = {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  };

  if (token) {
    headers['Authorization'] = `Bearer ${token}`;
  }

  const config = {
    method,
    headers,
    credentials: 'include'
  };

  if (body) {
    config.body = JSON.stringify(body);
  }

  return fetch(`http://localhost:8000/api/v1${url}`, config)
    .then(res => res.json())
    .catch(err => console.error(err));
};

// Usar:
const courses = await apiCall('/course/list', 'GET');
```

## Rutas Públicas (Sin Autenticación)

```
POST   /api/v1/public/auth/login
POST   /api/v1/public/auth/login/redirect
GET    /api/v1/public/banners/list
GET    /api/v1/public/events/list
GET    /api/v1/public/course/list
POST   /api/v1/public/registerAcademyUser
... y todas en el grupo /public
```

## Rutas Protegidas (Con Autenticación Required)

```
GET    /api/v1/course/list-marketplace/{id}
GET    /api/v1/course/{slug}
GET    /api/v1/course/temary/get-all-class/{slug}
GET    /api/v1/user/info
POST   /api/v1/user/update
POST   /api/v1/payments/save-payment
... y todas las demás que no están en /public
```

## Verificación

1. **Revisa la consola del navegador** (F12 → Network):
   - El request debe tener el header `Authorization: Bearer ...`
   - La respuesta debe ser 200, NO 401

2. **En Postman/Insomnia**:
   - Tab "Auth" → Type: "Bearer Token"
   - Pega el `access_token` obtenido en el login
   - O manualmente en Headers: `Authorization: Bearer <token>`

3. **En el backend** (Laravel logs):
   - `storage/logs/laravel.log` debe mostrar requests exitosos

## Troubleshooting

| Problema | Solución |
|----------|----------|
| "Unauthenticated." | Verifica que envíes el header `Authorization: Bearer <token>` |
| CORS error | Asegúrate que el frontend esté en `localhost:3000` o `localhost:3001` |
| Token expirado | Se expira en 1 hora. Realiza login nuevamente |
| 405 Method Not Allowed | Verifica que uses GET/POST/PUT/DELETE correctamente |

## Configuración para Producción

En `.env`, cambia la URL de origen permitida:
```env
FRONTEND_URL=https://tudominio.com
```

Y actualiza `app/Http/Middleware/Cors.php`:
```php
$allowed_origins = [
    env('FRONTEND_URL'),
    'https://www.tudominio.com'
];
```
